import { HttpService } from '@nestjs/axios';
import { BadRequestException, forwardRef, HttpException, HttpStatus, Inject, Injectable, NotFoundException } from '@nestjs/common';
import { firstValueFrom, lastValueFrom } from 'rxjs';
import { EventEmitter2 } from '@nestjs/event-emitter';
import { User } from 'src/shared/users/entities/user.entity';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { CreateChargeDto, CreateCreditChargeDto } from './dto/create-asaas-dto';
import { Status } from 'src/shared/enums/enum';
import { Order } from '../orders/entities/order.entity';
import { OrdersService } from '../orders/orders.service';
import { AxiosResponse } from 'axios';
import { CryptoUtil } from 'src/utils/encrypt';
import { Course } from '../courses/entites/courses.entity';
import { Course_payment } from '../course_payment/entities/course_payment.entity';
import { Creditcard } from '../creditcard/entities/creditcard.entity';

@Injectable()
export class AsaasService {
  private readonly baseURL = process.env.ASAAS_BASE_URL;
  private readonly headers = {
    accept: 'application/json',
    access_token: process.env.ASAAS_API_KEY,
    'content-type': 'application/json',
  };

  constructor(
    private readonly http: HttpService,
    private readonly eventEmitter: EventEmitter2,

    @Inject(forwardRef(() => OrdersService))
    private readonly ordersService: OrdersService,

    @InjectRepository(User)
    private readonly userRepository: Repository<User>,

    @InjectRepository(Creditcard)
    private readonly creditcardRepository: Repository<Creditcard>,


    @InjectRepository(Order)
    private readonly orderRepositoty: Repository<Order>,

    @InjectRepository(Course_payment)
    private readonly course_paymentRepository: Repository<Course_payment>,

  ) {}

  async handlePaymentWebhook(payload: any): Promise<void> {
    const { event, payment } = payload;

    if (!event || !payment) {
      throw new BadRequestException('Payload do webhook inválido.');
    }

    if (event === 'PAYMENT_RECEIVED'  || event === 'PAYMENT_CONFIRMED') {
      this.eventEmitter.emit('payment.confirmed', payment);

      const id = payment.id;
      const existCourse = await this.course_paymentRepository.findOne({
        where: {
          asaasPaymentId: id,
        },
      });

      const existOrder = await this.orderRepositoty.findOne({
        where: {
          paymentId: id,
        },
      });

      if (existCourse) {
        await this.course_paymentRepository.update(existCourse.id, {
          status: "PAID",
        });
        this.eventEmitter.emit('course.payment.confirmed', {
            orderId: existCourse.id,
          })
      }
      if (existOrder) {
        await this.orderRepositoty.update(existOrder.id, {
          status: Status.FINISHED
        });
        this.eventEmitter.emit('order.payment.confirmed', {
          orderId: existOrder.id,
        });
      }

    }
    
  }

  async createPixCharge(createChargeDto: CreateChargeDto): Promise<{ paymentId: string; qrCode: string; copyPaste: string }> {
    const { user, value, orderId } = createChargeDto;
    console.log("value " + value)
    const customerId = await this.getOrCreateCustomer(user);
    console.log(customerId)

    const paymentPayload = {
      customer: customerId,
      billingType: 'PIX',
      value,
      dueDate: new Date().toISOString().slice(0, 10),
      description: `Pedido #${orderId}`,
      externalReference: String(orderId),
    };
    console.log("AQUI2");
    try {
      const response = await firstValueFrom(
        this.http.post(`${this.baseURL}/lean/payments`, paymentPayload, { headers: this.headers }),
      )
      console.log("AQUI3");
      const paymentId = response.data.id;

      const qrCodePayload = await this.getPixQrCode(paymentId);
      return {
        paymentId,
        qrCode: qrCodePayload.encodedImage,
        copyPaste: qrCodePayload.payload,
      };
    } catch (error) {
      console.error('Erro ao criar cobrança no Asaas:', error.response?.data || error.message);
      throw new BadRequestException('Não foi possível gerar a cobrança de pagamento.');
    }
  }

  async createPixChargeForCourse(createChargeDto: CreateChargeDto): Promise<{ paymentId: string; qrCode: string; copyPaste: string }> {
    const { user, value, orderId } = createChargeDto;
    console.log("value " + value)
    const customerId = await this.getOrCreateCustomer(user);

    const paymentPayload = {
      customer: customerId,
      billingType: 'PIX',
      value,
      dueDate: new Date().toISOString().slice(0, 10),
      description: `Curso com ID #${orderId}`,
      externalReference: String(orderId),
    };
    console.log("AQUI2 - Payload:", JSON.stringify(paymentPayload, null, 2)); // Veja o que está enviando

    try {
      const response = await firstValueFrom(
        this.http.post(`${this.baseURL}/payments`, paymentPayload, { headers: this.headers }),
      );
      
      console.log("AQUI3 - Sucesso");
      const paymentId = response.data.id;
      console.log(paymentId)

      const qrCodePayload = await this.getPixQrCode(paymentId);
      return {
        paymentId,
        qrCode: qrCodePayload.encodedImage,
        copyPaste: qrCodePayload.payload,
      };
      
    } catch (error) {
      console.log("--- ENTROU NO CATCH ---");
      
      // Verifica se o erro veio da resposta da API (Ex: 400, 401, 500)
      if (error.response) {
        console.error('Status do Erro:', error.response.status);
        console.error('Dados do Erro (Asaas):', JSON.stringify(error.response.data, null, 2));
      } else {
        // Erro de rede ou código
        console.error('Erro sem resposta (Rede/Código):', error.message);
      }

      throw new BadRequestException('Não foi possível gerar a cobrança de pagamento.');
    }
  }

  async createCreditCardPayment(createChargeDto: CreateCreditChargeDto) {
    const { user, value, orderId, creditCard } = createChargeDto;
    const customerId = await this.getOrCreateCustomer(user);
    const payload ={
      customer: customerId,
      billingType: 'CREDIT_CARD',
      creditCard: {
        holderName: CryptoUtil.decrypt(creditCard.credit_card_holder_name),
        number: CryptoUtil.decrypt(creditCard.credit_card_number),
        expiryMonth: CryptoUtil.decrypt(creditCard.credit_card_expiry_month),
        expiryYear: CryptoUtil.decrypt(creditCard.credit_card_expiry_year),
        ccv: CryptoUtil.decrypt(creditCard.ccv),
      },
      value,
      dueDate: new Date().toISOString().slice(0, 10),
      description: `Pedido #${orderId}`,
      externalReference: String(orderId),
    };

    try {
      const response = await firstValueFrom(
        this.http.post(`${this.baseURL}/payments`, payload, { headers: this.headers }),
      );
      const paymentId = response.data.id;
      return { paymentId };
    }catch (error) {
      const asaasError = error.response?.data?.errors?.[0];
      const errorMessage = asaasError?.description || 'Não foi possível processar o pagamento com cartão de crédito.';
      const statusCode = error.response?.status || HttpStatus.BAD_REQUEST;

      console.error('Erro ao criar pagamento com cartão de crédito no Asaas:', error.response?.data || error.message);
      
      // Lança uma exceção com a mensagem de erro específica do Asaas e o status code original.
      throw new HttpException(errorMessage, statusCode);
    }
  }
  async createCreditCardPaymentCourse(CreateCreditChargeDto: CreateCreditChargeDto) {
    const {user, value, orderId, creditCard } = CreateCreditChargeDto;
    const customerId = await this.getOrCreateCustomer(user);
    const payload ={
      customer: customerId,
      billingType: 'CREDIT_CARD',
      creditCard: {
        holderName:CryptoUtil.decrypt(creditCard.credit_card_holder_name),
        number: CryptoUtil.decrypt(creditCard.credit_card_number),
        expiryMonth: CryptoUtil.decrypt(creditCard.credit_card_expiry_month),
        expiryYear: CryptoUtil.decrypt(creditCard.credit_card_expiry_year),
        ccv: CryptoUtil.decrypt(creditCard.ccv),
      },
      value,
      dueDate: new Date().toISOString().slice(0, 10),
      description: `Curso com ID#${orderId}`,
      externalReference: String(orderId),
    };

    try {
      const response = await firstValueFrom(
        this.http.post(`${this.baseURL}/payments`, payload, { headers: this.headers }),
      );
      const paymentId = response.data.id;
      return { paymentId };
    }catch (error) {
      const asaasError = error.response?.data?.errors?.[0];
      const errorMessage = asaasError?.description || 'Não foi possível processar o pagamento com cartão de crédito.';
      const statusCode = error.response?.status || HttpStatus.BAD_REQUEST;

      console.error('Erro ao criar pagamento com cartão de crédito no Asaas:', error.response?.data || error.message);

      // Lança uma exceção com a mensagem de erro específica do Asaas e o status code original.
      throw new HttpException(errorMessage, statusCode);
    }
  }

  // ### Funções Auxiliares ###

  private async getOrCreateCustomer(user: User) {
  if (user.asaas_customer_code) {
    return user.asaas_customer_code;
  }

  try {
    // Busca cliente existente
    const { data } = await firstValueFrom(
      this.http.get(`${this.baseURL}/customers`, {
        headers: this.headers,
        params: { cpfCnpj: user.document },
      }),
    );

    if (data.totalCount > 0) {
      const customerId = data.data[0].id;
      await this.userRepository.update(user.id, { asaas_customer_code: customerId });
      return customerId;
    }

    // Cria novo cliente
    const { data: newCustomer } = await firstValueFrom(
      this.http.post(
        `${this.baseURL}/customers`,
        {
          name: user.user_name,
          cpfCnpj: user.document,
          mobilePhone: user.cellphone,
        },
        { headers: this.headers },
      ),
    );

    await this.userRepository.update(user.id, { asaas_customer_code: newCustomer.id });
    return newCustomer.id;

  } catch (error) {
    const status = error.response?.status || 500;
    const message = error.response?.data || error.message || 'Erro ao integrar com Asaas';
    console.error('Erro Asaas:', message);
    throw new HttpException(message, status);
  }
}

  async getPixQrCode(paymentId: string): Promise<any> {
    const { data } = await firstValueFrom(
      this.http.get(`${this.baseURL}/payments/${paymentId}/pixQrCode`, { headers: this.headers }),
    );
    return data;
  }

  async fakePayment(userId: number, orderId: number) {
    const order = await this.orderRepositoty.findOne({
      where: {
        id: orderId,
      },
      relations: ['user']
    })

    if (!order) {
      throw new NotFoundException("Ops. Nenhum pedido pendente de pagamento localizado para esse usuário");
    }

    order.status = Status.PAID
    await this.orderRepositoty.save(order);
    this.eventEmitter.emit('payment.confirmed', {
      orderId: order.id,
    });

    // this.eventEmitter.emit('notify.adminStore', {cellphone: order?.store.cellphone, message: "Há um novo pedido pago aguardando para ser enviado"});

    return { message: "Pagamento feito"};
  }


  async createSubscription(userId: any) {
    try {

      const user = await this.userRepository.findOne({ 
        where : { id: userId }
       })

      if (!user) {
        throw new NotFoundException('Usuário não encontrado para criar assinatura');
      }

      
      const customerId = await this.getOrCreateCustomer(user);

      const assinaturaCriada = await this.http.post(
        `${this.baseURL}/subscriptions`,
        {
          customer: customerId,
          billingType: "PIX",
          nextDueDate: "2025-10-27",
          value: 10.0,
          cycle: `WEEKLY`,
          description: `Assinatura teste SEMANAL`,
        },
        {
          headers : this.headers
        },
      );

      const response: AxiosResponse = await lastValueFrom(assinaturaCriada);
      const data = response.data; // agora definido
      const asaasSubscriptionId = data.id;
      // exemplo: data.id será o id da assinatura no Asaas

      console.log("RESPONSE ASASS ", response);
      
      return data;
    } catch (error) {
      throw new HttpException(`Erro ao criar assinatura no Asaas: `, error);
    }
  }


  async paySubscription(userId: any) {
  try {
    // 1️⃣ Busca o usuário
    const creditCard = await this.creditcardRepository.findOne({ where: { user: userId } });
    if (!creditCard) {
      throw new NotFoundException('Cartão de crédito não encontrado para pagamento da assinatura');
    }


    const user= await this.userRepository.findOne({ where: { id: userId } });

    console.log("USER ", user);
    

    if (!user) {
      throw new NotFoundException('Usuário não encontrado para pagamento da assinatura');
    }

    const customerId = await this.getOrCreateCustomer(user);

    // 3️⃣ Cria o pagamento da assinatura no Asaas

      const decryptedCreditCard = {
        credit_card_holder_name: CryptoUtil.decrypt(creditCard.credit_card_holder_name),
        credit_card_number: CryptoUtil.decrypt(creditCard.credit_card_number),
        credit_card_expiry_month: CryptoUtil.decrypt(creditCard.credit_card_expiry_month),
        credit_card_expiry_year: CryptoUtil.decrypt(creditCard.credit_card_expiry_year),
        ccv: CryptoUtil.decrypt(creditCard.ccv),
      };

    const pagamentoCriado = await this.http.post(
      `${this.baseURL}/payments`,
      {
        customer: customerId,     // cliente Asaas
        billingType: "CREDIT_CARD",            // tipo de pagamento
        value: 10.0,
        dueDate: "2025-11-08",                  // data de vencimento
        description: "Pagamento da assinatura semanal",
        creditCard: {
          holderName: decryptedCreditCard.credit_card_holder_name,
          number: decryptedCreditCard.credit_card_number,
          expiryMonth: decryptedCreditCard.credit_card_expiry_month,
          expiryYear: decryptedCreditCard.credit_card_expiry_year,
          ccv: decryptedCreditCard.ccv,
        },
        creditCardHolderInfo: {
          name: user.user_name,
          email: user.email,
          cpfCnpj: user.document,
          postalCode: user.cep,
          addressNumber: user.house_number,
          phone: user.cellphone,
        },
      },
      { headers: this.headers },
    );

    const response: AxiosResponse = await lastValueFrom(pagamentoCriado);
    const data = response.data;

    console.log("PAGAMENTO REALIZADO: ", data);
    return data;
  } catch (error) {
    console.error("Erro ao pagar assinatura: ", error.response?.data || error.message);
    throw new HttpException('Erro ao processar pagamento no Asaas', HttpStatus.BAD_REQUEST);
  }
}


async getUserSubscriptions(userId: any) {
  try {
    // 1️⃣ Busca o usuário no banco
    const user = await this.userRepository.findOne({ where: { id: userId } });

    if (!user) {
      throw new NotFoundException('Usuário não encontrado');
    }

    // 2️⃣ Busca (ou cria) o customerId do Asaas
    const customerId = await this.getOrCreateCustomer(user);

    // 3️⃣ Chama o endpoint do Asaas para buscar assinaturas
    const response = await this.http.get(
      `${this.baseURL}/subscriptions?customer=${customerId}`,
      { headers: this.headers },
    );

    const result: AxiosResponse = await lastValueFrom(response);
    const subscriptions = result.data?.data || [];

    console.log("ASSINATURAS DO USUÁRIO:", subscriptions);

    // 4️⃣ Retorna a lista de assinaturas
    return subscriptions;
  } catch (error) {
    console.error('Erro ao buscar assinaturas no Asaas:', error.response?.data || error.message);
    throw new HttpException(
      'Erro ao buscar assinaturas do usuário no Asaas',
      HttpStatus.BAD_REQUEST,
    );
  }
}



}