import { Controller, Get, Post, Body, Patch, Param, Delete, UseGuards, UseInterceptors, UploadedFile, BadRequestException, ParseIntPipe } from '@nestjs/common';
// import { courseService } from './videos.service';

import { RequiredRoles } from 'src/shared/auth/required-roles.decorator';
import { AuthGuard } from 'src/shared/auth/auth.guard';
import { RoleGuard } from 'src/shared/auth/role/role.guard';
import { Roles } from 'src/shared/enums/enum';
import { FileInterceptor } from '@nestjs/platform-express';
import { diskStorage } from 'multer';
import * as multerS3 from 'multer-s3';
import { S3Client, PutObjectCommand } from '@aws-sdk/client-s3';
import { extname } from 'path';
import { UpdateCourseDto } from './dto/update-courses.dto';
import { CreateCourseDto } from './dto/create-courses.dto';
import { CourseService } from './courses.service';
import { CurrentUser } from 'src/shared/users/current.user.decorator';


@Controller('course')
export class CourseController {
  constructor(private readonly courseService: CourseService) {}

 @Post('/registro')
@UseInterceptors(FileInterceptor('file')) // 'file' é o nome do campo da imagem no Postman
async create(
  @Body() body: any, // Recebe 'any' porque o multipart transforma tudo em string
  @UploadedFile() file: Express.Multer.File,
) {
  let createCourseDto: CreateCourseDto;

  // 1. TRATAMENTO DO JSON
  // Se você enviar o JSON dentro de uma chave chamada 'data' (Recomendado)
  if (body.data) {
    try {
      createCourseDto = JSON.parse(body.data);
    } catch (error) {
      throw new BadRequestException('Formato JSON inválido no campo "data"');
    }
  } else {
    // Tenta usar o body direto (funciona apenas se não tiver arrays complexos)
    createCourseDto = body;
  }

  // 2. TRATAMENTO DO ARQUIVO (Upload)
  // Se um arquivo foi enviado, fazemos o upload e salvamos a URL no DTO
  if (file) {
    // Supondo que você tenha um método de upload no service
    // Se não tiver, você pode passar o 'file' como 2º argumento para o create
    const thumbnailUrl = await this.courseService.uploadImageToS3(file, file.originalname);
    
    // Atribui a URL ao campo correto do DTO (ex: thumbnail ou url)
    createCourseDto.thumbnail = thumbnailUrl; 
  }

  // 3. CRIAÇÃO
  return this.courseService.create(createCourseDto);
}


  @Patch(':courseId/class/:classId/complete')
  @UseGuards(AuthGuard)
  async completeLesson(
    @Param('courseId', ParseIntPipe) courseId: number,
    @Param('classId', ParseIntPipe) classId: number,
    @CurrentUser() user: any,
  ) {
    return this.courseService.completeLesson(courseId, classId, user.id);
  }

  @Get(':courseId/progress')
  @UseGuards(AuthGuard)
  async courseProgress(
    @Param('courseId', ParseIntPipe) courseId: number,
    @CurrentUser() user: any,
    ) {
    return this.courseService.courseProgress(user.id, courseId);
  }





  @Get('/user')
  @UseGuards(AuthGuard)
  async findAllByUser(@CurrentUser() user: any) {
    return this.courseService.findAllByUser(user.id);
  }

  @Get('/user/:id')
  @UseGuards(AuthGuard)
  async findOneByUser(@Param('id') courseId: number, @CurrentUser() user: any) {
    return this.courseService.findOneByUser(courseId, user.id);
  }




  @Get('/')
  findAll() {
    return this.courseService.findAll();
  }

  @Get('/restrito')
  @UseGuards(AuthGuard, RoleGuard)
  @RequiredRoles(Roles.ADMIN, Roles.ADMECOMMERCE)
  findAllForManagement() {
    return this.courseService.findAllForManagement();
  }

  @Get(':id')
  findOne(@Param('id') id: string) {
    return this.courseService.findOne(+id);
  }

  @Get(':id/restrito')
  @UseGuards(AuthGuard, RoleGuard)
  @RequiredRoles(Roles.ADMIN, Roles.ADMECOMMERCE)
  findOneForManagement() {
    return this.courseService.findOneForManagement();
  }


 @Patch('editar/:id') 
  @UseInterceptors(FileInterceptor('file'))
  async update(
    @Param('id') id: string,
    @Body() body: { data: string }, // <--- MUDANÇA 1: Recebe o wrapper 'data' como string
    @UploadedFile() file?: Express.Multer.File,
  ) {
    console.log("FILE CONTROLLER ", file);
    
    let parsedDto: UpdateCourseDto;

    // 1. FAZ O PARSE DA STRING JSON QUE VEIO DO FRONT
    try {
      parsedDto = JSON.parse(body.data);
    } catch (error) {
      throw new BadRequestException('Erro ao processar dados do formulário');
    }

    // 2. SE TIVER ARQUIVO, FAZ O UPLOAD E ATRIBUI AO DTO
    if (file) {
      const ext = extname(file.originalname);
      const fileName = `${Date.now()}-${Math.round(Math.random() * 1e9)}${ext}`;

      try {
         // Descomente e ajuste sua lógica de S3 aqui
         // const s3Url = await this.courseService.uploadImageToS3(file, fileName);
         
         // IMPORTANTE: O nome do campo no banco é 'thumbnail', não 'url'
         // parsedDto.thumbnail = s3Url; 
         
         // Para teste local se não tiver S3 ainda:
         parsedDto.thumbnail = fileName; 
         
      } catch (error) {
        throw new BadRequestException('Falha ao enviar imagem para o S3: ' + error);
      }
    }

    // 3. CHAMA O SERVICE PASSANDO O OBJETO JÁ CONVERTIDO
    return this.courseService.update(+id, parsedDto);
  }
  
  @Delete(':id')
  //@UseGuards(AuthGuard, RoleGuard)
  //@RequiredRoles(Roles.ADMIN, Roles.ADMECOMMERCE)
  remove(@Param('id') id: string) {
    return this.courseService.remove(+id);
  }


  @Get('/classes')
  // @UseGuards(AuthGuard, RoleGuard)
  // @RequiredRoles(Roles.ADMIN, Roles.ADMECOMMERCE)
  findCourseWithClasses() {
    return this.courseService.findCourseWithClasses();
  }

  @Get(':id/classes')
  // @UseGuards(AuthGuard, RoleGuard)
  // @RequiredRoles(Roles.ADMIN, Roles.ADMECOMMERCE)
  findCourseByIdWithClasses(@Param('id', ParseIntPipe) id: number) {
    return this.courseService.findCourseByIdWithClasses(id);
  }
}
