import {
  BadRequestException,
  ConflictException,
  ForbiddenException,
  Injectable,
  NotFoundException,
} from '@nestjs/common';
// import { CreateSignatureDto } from './dto/create-signature.dto';
import { InjectRepository } from '@nestjs/typeorm';
import { In, Repository } from 'typeorm';
// import { UpdateSignatureDto } from './dto/update-signature.dto';
import { diskStorage } from 'multer';
import * as multerS3 from 'multer-s3';
import { S3Client, PutObjectCommand } from '@aws-sdk/client-s3';
import * as sharp from 'sharp';

import { Course } from './entites/courses.entity';
import { CreateCourseDto } from './dto/create-courses.dto';
import { UpdateCourseDto } from './dto/update-courses.dto';
import { Class } from '../videos/entities/class.entity';
import { User } from 'src/shared/users/entities/user.entity';
import { Course_payment } from '../course_payment/entities/course_payment.entity';
import { CourseLessonProgress } from './entites/course_lesson_progress.entity';

@Injectable()
export class CourseService {
  constructor(
    @InjectRepository(Course)
    private courseRepository: Repository<Course>,

    @InjectRepository(Class)
    private classRepository: Repository<Class>,

    @InjectRepository(User)
    private userRepository: Repository<User>,

    @InjectRepository(Course_payment)
    private course_paymentRepository: Repository<Course_payment>,

    @InjectRepository(CourseLessonProgress)
    private courseLessonProgressRepository: Repository<CourseLessonProgress>,
  ) {}

  private s3 = new S3Client({
    region: process.env.AWS_S3_REGION!,
    credentials: {
      accessKeyId: process.env.AWS_ACCESS_KEY_ID!,
      secretAccessKey: process.env.AWS_SECRET_ACCESS_KEY!,
    },
  });

  async create(createCourseDto: CreateCourseDto) {
    const { classIds, ...courseData } = createCourseDto;

    // 1. Busca as aulas
    const classes = await this.classRepository.findBy({
      id: In(classIds),
    });

    // 2. Validação (se achou todas)
    if (classes.length !== classIds.length) {
      const foundIds = classes.map((c) => c.id);
      const notFoundIds = classIds.filter((id) => !foundIds.includes(id));
      throw new NotFoundException(
        `Video aulas não encontradas: ${notFoundIds.join(', ')}`,
      );
    }

    // 3. EXTRAI TODAS AS URLS
    // O .map cria um novo array contendo apenas o campo url de cada aula
    const allUrls = classes.map((classe) => classe.url);

    // 4. Cria o curso salvando o array de URLs
    const course = this.courseRepository.create({
      ...courseData,
      urls: allUrls, // O TypeORM converterá isso para string automaticamente
      classes: classes,
    });

    return await this.courseRepository.save(course);
  }

  async findAll() {
    // Usamos o QueryBuilder para poder adicionar um campo calculado (enrollments)
    const courses = await this.courseRepository
      .createQueryBuilder('course')
      .loadRelationCountAndMap(
        'course.enrollments', // Nome do novo campo que será criado no objeto de retorno
        'course.payments', // Relação na entidade Course que aponta para Course_payment
        'payment', // Alias para a tabela de pagamentos
        (qb) => qb.where('payment.status = :status', { status: 'PAID' }), // Condição para contar apenas matrículas confirmadas
      )
      .leftJoinAndSelect('course.classes', 'classes') // Inclui as aulas relacionadas
      .getMany();

    // A relação 'payments' não existe na entidade, vamos criá-la.
    // O TypeORM é inteligente o suficiente para usar a relação inversa do Course_payment.
    return courses;
  }

  async findAllByUser(CurrentUser: any) {
    const query = this.courseRepository
      .createQueryBuilder('course')
      .select([
        'course.id AS id',
        'course.title AS title',
        'course.description AS description',
        'course.category AS category',
        'course.thumbnail AS thumbnail',
        'course.level AS level',
        'course.price AS price',
      ])
      .where('course.status = :status', { status: 'Publicado' })
      .addSelect(
        'CASE WHEN payment.id IS NOT NULL THEN TRUE ELSE FALSE END',
        'hasAccess',
      )
      .leftJoin(
        'course_payment',
        'payment',
        'payment.courseId = course.id AND payment.userId = :userId AND payment.status = :paymentStatus',
      )
      // Adiciona os parâmetros para o JOIN aqui, junto com os do WHERE
      .setParameters({
        status: 'Publicado',
        userId: CurrentUser,
        paymentStatus: 'PAID',
      });

    return await query.getRawMany();
  }

  async findOneByUser(courseId: number, CurrentUser: any) {
    const course = await this.courseRepository.findOne({
      select: {
        id: true,
        title: true,
        description: true,
        category: true,
        status: true,
        thumbnail: true,
        level: true,
        price: true,
      },
      where: { id: courseId },
      relations: ['classes'],
    });

    if (!course) {
      throw new NotFoundException('Curso não encontrado');
    }
    const payment = await this.course_paymentRepository.findOne({
      where: {
        course: { id: courseId },
        user: { id: CurrentUser },
        status: 'PAID',
      },
    });

    if (!payment) {
      return { ...course, hasAccess: false };
    }

    return { ...course, hasAccess: true };
  }

  findAllForManagement() {
    return this.courseRepository.find();
  }

  async findOneForManagement() {
    return this.courseRepository.find();
  }

  async findOne(courseId: number) {
    const course = await this.courseRepository.findOne({
      select: {
        id: true,
        title: true,
        description: true,
        category: true,
        status: true,
        thumbnail: true,
        level: true,
        price: true,
        classes: {
          id: true,
          title: true,
          description: true,
          url: true,
        },
      },
      where: { id: courseId },
      relations: ['classes'],
    });
    if (!course) {
      throw new NotFoundException('Curso não encontrado');
    }

    // Mapeia o resultado para renomear 'classes' para 'lessons'
    const { classes, ...rest } = course;
    const response = { ...rest, lessons: classes || [] };

    return response;
  }

  async uploadImageToS3(
    file: Express.Multer.File,
    fileName: string,
  ): Promise<string> {
    const bucketName = process.env.AWS_S3_BUCKET!;

    //  const buffer = await sharp(file?.buffer)

    //         .toBuffer();

    const uploadParams = new PutObjectCommand({
      Bucket: bucketName,
      Key: fileName,
      Body: file.buffer,
      ContentType: file.mimetype,
    });

    console.log(uploadParams);

    try {
      await this.s3.send(uploadParams);
      return `https://${bucketName}.s3.${process.env.AWS_S3_REGION}.amazonaws.com/${fileName}`;
    } catch (err) {
      throw new BadRequestException('Erro ao enviar arquivo para o S3: ' + err);
    }
  }

  async update(courseId: number, updateCourseDto: UpdateCourseDto) {
    // 1. Separa os IDs das aulas do restante dos dados
    const { classIds, ...courseData } = updateCourseDto;

    // 2. Busca o curso antigo no banco
    const course = await this.courseRepository.findOne({
      where: { id: courseId },
      relations: ['classes'], // Carrega as relações atuais (opcional, mas seguro)
    });

    if (!course) {
      throw new NotFoundException(`Curso com ID ${courseId} não encontrado.`);
    }

    // 3. Lógica de Atualização das Aulas (Se o front enviou classIds)
    if (classIds && classIds.length > 0) {
      // Busca os objetos completos das aulas baseados nos IDs [19, 22]
      const classesEntities = await this.classRepository.findBy({
        id: In(classIds),
      });

      // Validação: Se enviou IDs, mas não achou nada (opcional)
      if (classesEntities.length !== classIds.length) {
        throw new NotFoundException('Algumas aulas informadas não existem.');
      }

      // --- O PULO DO GATO ---
      // Ao atribuir as novas entidades aqui, o TypeORM entende que deve
      // SUBSTITUIR as relações antigas na tabela pivô pelas novas.
      course.classes = classesEntities;

      // Atualiza o array de URLs (se você ainda estiver usando esse campo duplicado)
      course.urls = classesEntities.map((c) => c.url);
    }

    // 4. Atualiza os dados de texto (Title, Description, Price, etc)
    // O merge ignora o campo 'classes' pois já tratamos ele acima (ou ele não veio no DTO)
    this.courseRepository.merge(course, courseData);

    // 5. Salva tudo
    try {
      return await this.courseRepository.save(course);
    } catch (error) {
      console.error(error);
      throw new BadRequestException('Erro ao atualizar o curso.');
    }
  }

  async remove(courseId: number) {
    const course = await this.courseRepository.findOne({
      where: { id: courseId },
    });

    if (!course) {
      throw new NotFoundException(`Curso com ID ${courseId} não encontrado.`);
    }

    try {
      await this.courseRepository.remove(course);
    } catch (error) {
      console.log(error);
      throw new BadRequestException(
        'Erro ao tentar remover o curso. Erro',
        error,
      );
    }

    return { message: 'Curso removido com sucesso' };
  }

  /* ### Funções Auxiliares ### */

  async findCourseById(courseId: number) {
    const course = await this.courseRepository.findOneBy({ id: courseId });

    if (!course) {
      throw new NotFoundException('Curso não encontrado');
    }

    return course;
  }

  async findCourseWithClasses() {
    const course = await this.courseRepository.find({
      relations: ['classes'],
    });

    return course;
  }

  async findCourseByIdWithClasses(id: number) {
    const course = await this.courseRepository.find({
      where: { id },
      relations: ['classes'],
    });

    return course;
  }

  async completeLesson(courseId: number, classId: number, CurrentUser: any) {
    const iscompleted = await this.courseLessonProgressRepository.findOne({
      where: {
        user: { id: CurrentUser },
        course: { id: courseId },
        class: { id: classId },
        completed: true,
      },
    });
    if (iscompleted) {
      return { completed: true };
    }

    await this.courseLessonProgressRepository.save({
      course: { id: courseId },
      class: { id: classId },
      user: { id: CurrentUser },
      completed: true,
    });
    return { completed: true };
  }

  async courseProgress(CurrentUser: any, courseId: number) {
    // 1 — validar compra
    const course = await this.courseRepository
      .createQueryBuilder('course')
      .select([
        'course.id AS id',
        'course.title AS title',
        'course.description AS description',
      ])
      .leftJoin('course_payment', 'payment', 'payment.courseId = course.id')
      .where('course.status = :status', { status: 'Publicado' })
      .andWhere('course.id = :courseId', { courseId })
      .andWhere('payment.userId = :userId', { userId: CurrentUser })
      .andWhere('payment.status = :paymentStatus', { paymentStatus: 'PAID' })
      .getRawOne();

    if (!course) {
      throw new Error('Curso não encontrado ou não pago');
    }

    // 2 — progresso
    const totalLessons = await this.classRepository.count({
      where: { courses: { id: courseId } },
    });

    const completedLessons = await this.courseLessonProgressRepository.count({
      where: {
        user: { id: CurrentUser },
        course: { id: courseId },
        completed: true,
      },
    });

    const progress = Math.round((completedLessons / totalLessons) * 100);

    return {
      ...course,
      totalLessons,
      completedLessons,
      progress,
      finished: completedLessons === totalLessons,
    };
  }
}
