import { ConflictException, Injectable, NotFoundException } from '@nestjs/common';
import { CreateCreditcardDto } from './dto/create-creditcard.dto';
import { UpdateCreditcardDto } from './dto/update-creditcard.dto';
import { User } from 'src/shared/users/entities/user.entity';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { Creditcard } from './entities/creditcard.entity';
import { CryptoUtil } from 'src/utils/encrypt';

@Injectable()
export class CreditcardService {
  constructor(
    @InjectRepository(Creditcard)
    private creditcardRepository: Repository<Creditcard>,

    @InjectRepository(User)
    private usersRepository: Repository<User>,
  ) {}



  async create(createCreditcardDto: CreateCreditcardDto, currentUser: any) {
    const user = await this.usersRepository.findOneBy({ id: currentUser.id });
    if (!user) {
      throw new NotFoundException('Usuário não encontrado.');
    }

    const cardNumber = createCreditcardDto.credit_card_number;
    const lastFourDigits = cardNumber.slice(-4);

    
    const encryptedCardNumber = await CryptoUtil.encrypt(cardNumber);

    const cardExist = await this.creditcardRepository.findOneBy({ credit_card_number: encryptedCardNumber, user: { id: user.id } });
    if (cardExist) {
      throw new ConflictException("Cartão já cadastrado para este usuário");
    }

    const newCard = this.creditcardRepository.create({
      ...createCreditcardDto,
      last_four_digits: lastFourDigits,
      credit_card_holder_name: await CryptoUtil.encrypt(createCreditcardDto.credit_card_holder_name),
      credit_card_number: encryptedCardNumber,
      credit_card_expiry_month: await CryptoUtil.encrypt(createCreditcardDto.credit_card_expiry_month),
      credit_card_expiry_year: await CryptoUtil.encrypt(createCreditcardDto.credit_card_expiry_year),
      ccv: await CryptoUtil.encrypt(createCreditcardDto.ccv),
      user: user,
    });

    return this.creditcardRepository.save(newCard);
  }

  async findAll(currentUser: any) {
    const existCard = await this.creditcardRepository.find({
      where: { user: { id: currentUser.id } },
    });

    if(!existCard){
      throw new NotFoundException("Cartão nenhum cartão encontrado");
    }
    return existCard;
  }

  async findLastCardRegister(currentUser: any) {
    const existCard = await this.creditcardRepository.findOne({
      where: { user: { id: currentUser.id } },
      order: { id: 'DESC' },
    });

    if(!existCard){
      throw new NotFoundException("Nenhum cartão encontrado");
    }
    return existCard;
  }

  update(id: number, updateCreditcardDto: UpdateCreditcardDto) {
    const existCard = this.creditcardRepository.findOneBy({ id });
    if(!existCard){
      throw new NotFoundException("Cartão não encontrado");
    }
    return this.creditcardRepository.update(id, updateCreditcardDto);
  }

  remove(id: number) {
    const existCard = this.creditcardRepository.findOneBy({ id });
    if(!existCard){
      throw new NotFoundException("Cartão não encontrado");
    }try {
      return this.creditcardRepository.delete(id);
    } catch (error) {
      throw new NotFoundException("Erro ao tentar deletar o cartão");
    }
  }
}
