import { BadRequestException, Injectable, NotFoundException } from '@nestjs/common';
import { CreateCupomDto } from './dto/create-cupom.dto';
import { UpdateCupomDto } from './dto/update-cupom.dto';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, ReturnDocument } from 'typeorm';
import { Cupom } from './entities/cupom.entity';
import { Order } from '../orders/entities/order.entity';
import { OrderProduct } from '../orders/entities/order_product.entity';

@Injectable()
export class CupomService {

  constructor(
    @InjectRepository(Cupom)
    private cupomRepository: Repository<Cupom>,
    @InjectRepository(Order)
    private orderRepository: Repository<Order>,
    @InjectRepository(OrderProduct)
    private orderProductRepository: Repository<OrderProduct>,
  ){}

  async create(createCupomDto: CreateCupomDto) {
    const exist = await this.cupomRepository.findOne({
      where: {
        codigo: createCupomDto.codigo
      }
    })
    if (exist) {
      throw new BadRequestException('Um cupom com este código já existe.');
    } else {
      const cupom = this.cupomRepository.create(createCupomDto);
      await this.cupomRepository.save(cupom);
      return cupom;
    }
  }

  async descont(orderId: number, codigo: string) {
    const order = await this.orderRepository.findOne({
      where: { id: orderId },
    });
    if (!order) {
      throw new NotFoundException('Pedido não encontrado.');
    }
    
    const cupom = await this.cupomRepository.findOne({
      where: { codigo: codigo }
    });
    if (!cupom) {
      throw new NotFoundException('Cupom inválido ou não encontrado.');
    }

    if (order.descont) {
      throw new BadRequestException('Um cupom já foi aplicado a este pedido. Remova o cupom existente antes de aplicar um novo.');
    }

    order.cupom = cupom;
    order.descont = true;
    
    await this.orderRepository.save(order);
    
    return order;

  }

  async removeDescont(orderId: number) {
    const order = await this.orderRepository.findOne({
      where: { id: orderId },
    });

    if (!order) {
      throw new NotFoundException('Pedido não encontrado.');
    }

    if (!order.descont) {
      throw new BadRequestException('Nenhum desconto foi aplicado a este pedido.');
    }

    
    order.cupom = null as any;
    order.descont = false;
    
    await this.orderRepository.save(order);
    return order;
  }

  findAll() {
    return this.cupomRepository.find();
  }

  findOne(id: number) {
    return `This action returns a #${id} cupom`;
  }

  update(id: number, updateCupomDto: UpdateCupomDto) {
    return this.cupomRepository.update(id, updateCupomDto);
  }

  remove(id: number) {
    return this.cupomRepository.delete(id);
  }
}
