import { Body, Controller, Get, Param, ParseEnumPipe, Patch, Query, UseGuards } from '@nestjs/common';
import { OrdersService } from './orders.service';
import { Roles, Status } from 'src/shared/enums/enum';
import { RoleGuard } from 'src/shared/auth/role/role.guard';
import { AuthGuard } from 'src/shared/auth/auth.guard';
import { RequiredRoles } from 'src/shared/auth/required-roles.decorator';
import { CurrentUser } from 'src/shared/users/current.user.decorator';
import { CheckoutDto } from './dto/checkout.dto';

@Controller('pedidos')
export class OrdersController {
  constructor(private readonly ordersService: OrdersService) {}
  
  @Get('/')
  @UseGuards(AuthGuard)
  findAllByUser(@CurrentUser() currentUser: any) {
    return this.ordersService.findAllByUser(currentUser.id);
  }

  @Get(':orderId/detalhe-pedido')
  @UseGuards(AuthGuard)
  findAllProductsByOrderId(@Param('orderId') orderId: string, @CurrentUser() currentUser: any) {
    return this.ordersService.findAllProductsByOrderId(currentUser.id, +orderId)
  }

  @Get('/todos')
  @UseGuards(AuthGuard, RoleGuard)
  @RequiredRoles(Roles.ADMIN, Roles.ADMECOMMERCE)
  findAll() {
    return this.ordersService.findAll();
  } 

  @Get('/todos-por-status')
  @UseGuards(AuthGuard, RoleGuard)
  @RequiredRoles(Roles.ADMIN, Roles.ADMECOMMERCE)
  findAllByStatus(@Query('status', new ParseEnumPipe(Status)) status: Status) {
    return this.ordersService.findAllByStatus(status);
  }

  @Get(':orderId/produtos')
  @UseGuards(AuthGuard, RoleGuard)
  @RequiredRoles(Roles.ADMIN, Roles.ADMECOMMERCE)
  restrictFindAllProductsByOrderId(@Param('orderId') orderId: string) {
    return this.ordersService.restrictFindAllProductsByOrderId(+orderId);
  }

  @Get('/qrcode')
  @UseGuards(AuthGuard)
  createQRcode(@CurrentUser() currentUser: any){
    return this.ordersService.createQRcode(currentUser.id);
  }

  @Get('/status/:paymentId')
  @UseGuards(AuthGuard)
  statusPaied(@Param('paymentId') paymentId: string){
    return this.ordersService.statusPaied(paymentId);
  }



  @Patch('/finalizar/')
  @UseGuards(AuthGuard)
  checkout(@CurrentUser() currentUser: any, @Body() chechkoutDto: CheckoutDto) {
    return this.ordersService.checkout(currentUser.id, chechkoutDto);
  }

  @Patch(':orderId')
  @UseGuards(AuthGuard, RoleGuard)
  @RequiredRoles(Roles.ADMIN, Roles.ADMECOMMERCE)
  changeOrderStatusById(@Param('orderId') orderId: string, @Query('status', new ParseEnumPipe(Status)) status: Status) {
    return this.ordersService.changeOrderStatusById(+orderId, status);
  }
}

