import { BadRequestException, Injectable, NotFoundException } from '@nestjs/common';
import { AddItemFromCartDto } from './dto/add-item-from-cart-.dto';
import { RemoveItemFromCartDto, UpdateShoppingCartDto } from './dto/update-shopping_cart.dto';
import { Order } from '../orders/entities/order.entity';
import { InjectRepository } from '@nestjs/typeorm';
import { In, Repository } from 'typeorm';
import { Status } from 'src/shared/enums/enum';
import { UsersService } from 'src/shared/users/users.service';
import { OrderTransactions } from '../orders/entities/order_transaction.entity';
import { ProductsService } from '../products/products.service';
import { OrderProduct } from '../orders/entities/order_product.entity';
import { Cupom } from '../cupom/entities/cupom.entity';

@Injectable()
export class ShoppingCartService {
  constructor (
    @InjectRepository(Order)
    private readonly shoppingCartRepository: Repository<Order>,

    @InjectRepository(OrderTransactions)
    private readonly orderTransactionRepository: Repository<OrderTransactions>,    

    @InjectRepository(OrderProduct)
    private readonly orderProductRepository: Repository<OrderProduct>,

    @InjectRepository(Cupom)
    private readonly cupomRepository: Repository<Cupom>,


    private readonly usersService: UsersService,
    private readonly productsService: ProductsService

  ) {}
  async addItemFromCart(currentUser: any, addItemFromCartDto: AddItemFromCartDto) {
    const product = await this.productsService.findProductById(addItemFromCartDto.productId);
    const cart = await this.getOrCreateShoppingCart(currentUser.id);
    const orderTransactionExist = await this.orderTransactionRepository.findOne({
      where: {
        order: {id: cart.id},
      }
    })

    if (orderTransactionExist && (orderTransactionExist.status === Status.PENDING_PAYMENT || orderTransactionExist.status === Status.PAID)) {
      throw new BadRequestException ("Ops você ainda tem pedido em andamento")
    }
    const orderItemExist = await this.orderProductRepository.findOne({
      where: {
        order: {id: cart.id},
        product: product,
      },
    })

    if (product.quantity < addItemFromCartDto.quantity) {
      throw new BadRequestException(`Quantidade do produto solicitado está indisponível. Apenas ${product.quantity} disponíveis no momento. Quantidade solicitada ${addItemFromCartDto.quantity}`);
    }

    if (orderItemExist) {
      Object.assign(orderItemExist, (orderItemExist.quantity += addItemFromCartDto.quantity));
      if(product.quantity < orderItemExist.quantity) {
        throw new BadRequestException(`Quantidade do produto solicitado está indisponível. Apenas ${product.quantity} disponíveis no momento. Quantidade solicitada ${addItemFromCartDto.quantity}`);
      }
      await this.orderProductRepository.save(orderItemExist);
    }else {
      await this.orderProductRepository.save(
        this.orderProductRepository.create({
          order: cart,
          product,
          quantity: addItemFromCartDto.quantity,
          unit_price: product.price,
        })
      );
    }

    await this._recalculateCartTotalsAndUpdateStoreId(cart.id)

    return { message: "Produto adicionado ao carrinho"};
  }

async getCartContents(currentUser: any) {
    const cart = await this.shoppingCartRepository.findOne({
      where: {
        user: { id: currentUser.id },
        status: In([Status.OPEN, Status.PENDING_PAYMENT, Status.PAID]),
      },
      relations: ['cupom']
    });

    if (!cart) {
      return { message: 'Seu carrinho está vazio' };
    }

    const itemsOnCart = await this.orderProductRepository.find({
      where: { order: { id: cart.id } },
      relations: ['product'],
    });

    // 1. IMPORTANTE: Recalcula o subtotal somando os itens.
    // Isso evita o bug de aplicar desconto sobre desconto se o usuário der F5.
    const subtotal = itemsOnCart.reduce((sum, item) => {
        return sum + (Number(item.product.price) * item.quantity);
    }, 0);

    let finalAmount = subtotal;
    let discountVal = 0;

    // 2. Verifica se existe cupom e calcula
    if (cart.cupom) {
        const percentage = Number(cart.cupom.desconto); 
        
        if (!isNaN(percentage) && percentage > 0) {
            discountVal = (subtotal * percentage) / 100; // Calcula sobre o subtotal limpo
            finalAmount = subtotal - discountVal;
        }
    }

    // 3. Atualiza e SALVA no banco de dados
    // Só salvamos se o valor for diferente para não estressar o banco à toa
    if (Number(cart.total_amount) !== finalAmount) {
        cart.total_amount = finalAmount;
        await this.shoppingCartRepository.save(cart);
        console.log(`Valor atualizado no banco: De ${subtotal} para ${finalAmount}`);
    }

    const formattedItems = itemsOnCart.map((item) => ({
      id: item.product.id,
      name: item.product.name,
      description: item.product.description,
      image: item.product.image,
      quantity: item.quantity,
      price: item.product.price,
      quantity_in_stock: item.product.quantity,
    }));

    return {
      order: {
        id: cart.id,
        status: cart.status,
        total_items: cart.total_items,
        cupom: cart.cupom,
        subtotal: subtotal,             
        total_amount: finalAmount,      
        discount_applied: discountVal,  
        items: formattedItems,
      },
    };
}

  async countTotalItemsInOpenOrder(currentUser: any) {
    const user = await this.usersService.findUserById(currentUser.id);
    const cart = await this.findOpenOrderByUser(user.id);

    let totalItems;
    if (!cart || !cart.order_product) {
      totalItems = 0;
    } else {
      totalItems = cart.order_product.reduce((total, item) => total + item.quantity, 0);
    }

    return {total_quantity: totalItems};
  }

  async updateItemQuantityInCart(currentUser: any, updateShoppingCartDto: UpdateShoppingCartDto ) {
    const product = await this.productsService.findProductById(updateShoppingCartDto.productId);
    const cart = await this.getOrCreateShoppingCart(currentUser.id);

    const orderItemExist = await this.orderProductRepository.findOne({
      where: {
        order: {id: cart.id},
        product: { id: product.id}
      }
    })

    if (!orderItemExist) {
      throw new NotFoundException("Ops. Não foi possível localizar o produto no carrinho");
    }

    Object.assign(orderItemExist, orderItemExist.quantity = updateShoppingCartDto.quantity);
    await this.orderProductRepository.save(orderItemExist);

    await this._recalculateCartTotalsAndUpdateStoreId(cart.id)


    return { message: "Quantidade atualizada com sucesso"}; 
  } 

  async removeItemFromCart(currentUser: any, removeItemFromCartDto: RemoveItemFromCartDto) {
    const product = await this.productsService.findProductById(removeItemFromCartDto.productId);
    const cart = await this.getOrCreateShoppingCart(currentUser.id);

    const orderItemExist = await this.orderProductRepository.findOne({
      where: {
        order: {id: cart.id},
        product: { id: product.id },
      }
    })
    console.log(cart.id);
    console.log(product);
    console.log(orderItemExist);

    if (!orderItemExist) {
      throw new NotFoundException("Ops. Não foi possível localizar o produto no carrinho");
    }

    await this.orderProductRepository.remove(orderItemExist);

    await this._recalculateCartTotalsAndUpdateStoreId(cart.id)
    
    return { message: "Produto removido do carrinho"};    

  }

  /* Funções Auxiliares */

  async getOrCreateShoppingCart(userId: number) {
    await this.usersService.findUserById(userId);

    let cart = await this.findOpenOrderByUser(userId);
    if (!cart) {      
      const newCart = await this.shoppingCartRepository.save(
        this.shoppingCartRepository.create({user: { id: userId }})
      );

      await this.orderTransactionRepository.save(
        this.orderTransactionRepository.create({
          order: newCart,
          status: Status.OPEN
        })
      )

      cart = newCart;
    }

    return cart;
  }

  async findOpenOrderByUser(userId: number) { 
    return await this.shoppingCartRepository.findOne({ 
      where: { 
        user: { id: userId}, 
        status: Status.OPEN, 
      },
      relations: ['order_product']
    }); 
  }

  private async _recalculateCartTotalsAndUpdateStoreId(cartId: number) {
    const cart = await this.shoppingCartRepository.findOneBy({ id: cartId });
    
    if (!cart) {
      console.error(`Tentativa de recalcular totais para o carrinho ID ${cartId}, que não foi encontrado.`);
      return;
    }

    const totals = await this.orderProductRepository
      .createQueryBuilder('orderProduct')
      .innerJoin('orderProduct.product', 'product')
      .select('SUM(orderProduct.quantity)', 'total_items')
      .addSelect('SUM(orderProduct.quantity * orderProduct.unit_price)', 'sub_total')
      .where('orderProduct.orderId = :cartId', { cartId })
      .getRawOne();

    await this.shoppingCartRepository.update(cartId, {
      total_items: Number(totals.total_items) || 0,
      total_amount: totals.sub_total || 0,
    });
  }
  // async findOtherOrdersByUser(userId: number) {
  //   const orderExist = await this.shoppingCartRepository.findOne({
  //     where: [
  //       { user: { id: userId}, status: Status.AWAITING_PAYMENT},
  //       { user: { id: userId}, status: Status.PENDING_APPROVAL},
  //     ]
  //   })

  //   if (orderExist) {
  //     throw new ConflictException("Ops. Você tem pedidos em aberto. Aguarde eles serem processados");
  //   }
  // }
}
