import { ApiProperty } from "@nestjs/swagger";
import { IsNotEmpty, IsNumber, IsString } from "class-validator";
import { Model } from "src/visoflex360/models/entities/model.entity";
import { User } from "src/shared/users/entities/user.entity";

export class CreateEquipamentDto {
    @ApiProperty({ description: 'Nome do equipamento', example: 'Porta Fast V5' })
    @IsString({message: "O nome do equipamento deve conter apenas texto"})
    @IsNotEmpty({message: "O nome do equipamento é obrigatório"})
    equipament_name?: string

    @ApiProperty({ description: 'ID interno do equipamento (o número do MAC)', example: '00:1A:2B:3C:4D:5E' })
    @IsString({ message: "O ID do equipamento deve conter apenas texto"})
    @IsNotEmpty({message: "O ID do equipamento é obrigatório"})
    internal_id: string;

    @ApiProperty({ description: 'Modelo do equipamento', type: Model, required: true, example: 1 })
    @IsNotEmpty({ message: "O modelo do equipamento é obrigatório" })
    @IsNumber({}, {message: "O modelo do equipamento deve ser um número" })
    model?: Model;

    @ApiProperty({ description: 'Setor do equipamento', example: 'Estoque de Produtos Acabados' })
    @IsString({message: "O setor do equipamento deve conter apenas texto"})
    @IsNotEmpty({message: "O setor do equipamento é obrigatório"})
    section?: string;

    @ApiProperty({ description: 'Id do usuário que está criando o equipamento', type: User, required: false, example: 1 })
    user?: User;
}

export class CreateEquipamentFirstConnectionDto {
    @ApiProperty({ description: 'Identificação interna do cliente', required: false })
    @IsString({ message: "A identificação interna deve ser uma string"})  
    internal_id: string;

    @ApiProperty({ description: 'MAC do equipamento', required: true, example: '44:33:D4:F5' })
    @IsNotEmpty({ message: "O MAC do equipamento é obrigatório"})
    @IsString({ message: "O MAC do equipamento deve ser uma string"})
    mac_address: string;

    @ApiProperty({ description: 'Nome do equipamento', required: true, example: "Porta de Testes v5"})
    @IsNotEmpty({ message: "O nome do equipamento é obrigatório"})
    @IsString({ message: "O nome do equipamento deve ser uma string válida"})
    equipament_name: string;

    @ApiProperty({ description: 'Setor em que o equipamento está', required: true, example: "Casa de Máquinas"})
    @IsNotEmpty({ message: "O nome do setor é obrigatório"})
    @IsString({ message: "O nome do setor deve ser uma string válida"})
    section: string;
    
    @IsNotEmpty({ message: 'O Código de Sincronização do usuário é obrigatório'})
    @IsString({ message: 'O código de Sincronização do usuário deve ser do tipo string'})
    sinc_code: string;

    @ApiProperty({ description: 'Modelo do equipamento', type: Model, required: true, example: 1 })
    @IsNotEmpty({ message: "O modelo do equipamento é obrigatório" })
    @IsNumber({}, { message: "O modelo do equipamento deve ser um número" })
    model: Model;
}