import { Controller, Get, Post, Body, Patch, Param, Delete, UseGuards, UploadedFile, UseInterceptors, Res } from '@nestjs/common';
import { Response } from 'express';
import { FileInterceptor } from '@nestjs/platform-express';
import { diskStorage } from 'multer';
import * as fs from 'fs';
import { extname, join } from 'path';
import { FilesService } from './files.service';
import { CreateFilesDto } from './dto/create-files.dto';
import { UpdateFilesDto } from './dto/update-files.dto';
import { RequiredRoles } from 'src/shared/auth/required-roles.decorator';

import { Roles } from 'src/shared/enums/enum';
import { AuthGuard } from 'src/shared/auth/auth.guard';
import { ApiBearerAuth, ApiOperation, ApiResponse, ApiTags } from '@nestjs/swagger';
import { CreateFolderDto } from './dto/create-path.dto';
import { plainToInstance } from 'class-transformer';
import { validateOrReject } from 'class-validator';
import { CurrentUser } from 'src/shared/users/current.user.decorator';
import { UpdatePathDto } from './dto/update-path.dto';

@ApiTags('Arquivos')
@Controller('arquivos')
export class FilesController {
  constructor(private readonly filesService: FilesService
  ) {}
  @Post('/registro')
@UseInterceptors(FileInterceptor('file', {
  storage: diskStorage({
    destination: (req, file, cb) => {
      const rawData = req.body['data'] ? JSON.parse(req.body['data']) : {};
      const pathFromUser = rawData.path || '';
      const uploadPath = './uploads';
      const fullPath = join(process.cwd(), uploadPath, pathFromUser);
      fs.mkdirSync(fullPath, { recursive: true });
      cb(null, fullPath);
    },
    filename: (req, file, cb) => {
      const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1e9);
      const ext = extname(file.originalname);
      cb(null, `${file.fieldname}-${uniqueSuffix}${ext}`);
    },
  }),
}))
async create(
  @UploadedFile() file: Express.Multer.File,
  @Body('data') data: string,
) {
  // transforma o JSON enviado como string
  const parsedData = JSON.parse(data);

  
  const dto = plainToInstance(CreateFilesDto, parsedData);

  await validateOrReject(dto); 

  return this.filesService.create(file, dto);
}



  @Post('/pasta')
  createFolder(@Body() createFolderDto: CreateFolderDto) {
    console.log("CONTROLLER CHAMADO");
    return this.filesService.createFolder(createFolderDto);
  }

  @Get('/pasta')
  @UseGuards(AuthGuard)
  getRaiz(@CurrentUser() currentUser: any) {
    return this.filesService.getRaiz(currentUser);
  }

  @Get('/pasta/:id')
  @UseGuards(AuthGuard)
  getRelation(@Param('id') id: number, @CurrentUser() currentUser: any){
    return this.filesService.getRelation(id, currentUser);
  }

  @Patch('/pasta/:id')
  @UseGuards(AuthGuard)
  editPath(@Param('id') id: number, @Body() updateFolderDto: UpdatePathDto) {
    return this.filesService.editPath(id, updateFolderDto)}




  @Delete('/delete/pasta/:id')
  async delete(@Param('id') id: number) {
  return this.filesService.deleteFolder(id);
}

@Get('donwload/:id')
async downloadFile(
  @Param('id') id: number,
  @Res() res: Response
) {
  const filePath = await this.filesService.getFilePath(id);

  // caminho absoluto
  const fullPath = join(filePath);

  return res.download(fullPath);
}

@Get(':id/view')
async viewImage(@Param('id') id: number, @Res() res: Response) {
  const filePath = await this.filesService.getFilePath(id);
  const fullPath = join(filePath);

  return res.sendFile(fullPath);
}





  @Get('/')
  @UseGuards(AuthGuard)
  findAll() {
    return this.filesService.findAll();
  }


  @Delete(':id')
  @UseGuards(AuthGuard)
  @RequiredRoles(Roles.ADMVISO360, Roles.ADMIN)
  remove(@Param('id') id: string) {
    return this.filesService.remove(+id);
  }
}
