import {
  Controller,
  Get,
  Post,
  Body,
  Patch,
  Param,
  Delete,
  UploadedFile,
  UseInterceptors,
  UseGuards,
} from '@nestjs/common';
import { PostService } from './post.service';
import { CreatePostDto } from './dto/create-post.dto';
import { UpdatePostDto } from './dto/update-post.dto';
import { FileInterceptor } from '@nestjs/platform-express';
import { memoryStorage } from 'multer';
import { CurrentUser } from 'src/shared/users/current.user.decorator';
import { AuthGuard } from 'src/shared/auth/auth.guard';
import { User } from 'src/shared/users/entities/user.entity';
import { CreateComentarioDto } from './dto/create-comentario.dto';
import { RequiredRoles } from 'src/shared/auth/required-roles.decorator';
import { Roles } from 'src/shared/enums/enum';
import { RoleGuard } from 'src/shared/auth/role/role.guard';

@Controller('post')
export class PostController {
  constructor(private readonly postService: PostService) {}

  @Post()
  @UseGuards(AuthGuard)
  @UseInterceptors(
    FileInterceptor('imagem', {
      storage: memoryStorage(),

      fileFilter: (req, file, cb) => {
        if (!file.mimetype.match(/\/(jpg|jpeg|png)$/)) {
          return cb(new Error('Apenas imagens JPG, JPEG ou PNG'), false);
        }
        cb(null, true);
      },
    }),
  )
  create(
    @Body('data') data: string,
    @UploadedFile() imagemUrl: Express.Multer.File,
    @CurrentUser() currentUser: User,
  ) {
    const createPostDto = JSON.parse(data);
    return this.postService.create(createPostDto, currentUser, imagemUrl);
  }

  @Post(':postId/comentario')
  @UseGuards(AuthGuard)
  createComentario(
    @Body() createComentarioDto: CreateComentarioDto,
    @CurrentUser() currentUser: User,
    @Param('postId') postId: string,
  ) {
    return this.postService.createComentario(
      createComentarioDto,
      currentUser,
      +postId,
    );
  }

  @Get()
  @UseGuards(AuthGuard, RoleGuard)
  @RequiredRoles(Roles.ADMIN, Roles.USER)
  findAll(@CurrentUser() currentUser: User) {
    return this.postService.findAll(currentUser);
  }

  @Get(':id/comentarios')
  getComentarios(@Param('id') id: string) {
    return this.postService.getComentarios(+id);
  }

  @Get(':id')
  @UseGuards(AuthGuard)
  findOne(@Param('id') id: string, @CurrentUser() currentUser: User) {
    return this.postService.findOne(+id, currentUser);
  }

  @Patch(':id/like')
  @UseGuards(AuthGuard)
  like(@Param('id') id: string, @CurrentUser() currentUser: User) {
    return this.postService.like(+id, currentUser);
  }
  @Patch(':id/deslike')
  @UseGuards(AuthGuard)
  deslike(@Param('id') id: string, @CurrentUser() currentUser: User) {
    return this.postService.deslike(+id, currentUser);
  }

  @Patch(':id')
  @UseGuards(AuthGuard)
  update(@Param('id') id: string, @Body() updatePostDto: UpdatePostDto) {
    return this.postService.update(+id, updatePostDto);
  }

  @Delete(':id')
  @UseGuards(AuthGuard)
  remove(@Param('id') id: string) {
    return this.postService.remove(+id);
  }

  @Delete(':idComentario/comentario')
  @UseGuards(AuthGuard)
  removeComentario(@Param('idComentario') idComentario: string) {
    return this.postService.removeComentario(+idComentario);
  }
}
