import { BadRequestException, Injectable } from '@nestjs/common';
import { CreatePostDto } from './dto/create-post.dto';
import { UpdatePostDto } from './dto/update-post.dto';
import { InjectRepository } from '@nestjs/typeorm';
import { Post } from './entities/post.entity';
import { In, Repository } from 'typeorm';
import { PutObjectCommand, S3Client } from '@aws-sdk/client-s3';
import { CreateComentarioDto } from './dto/create-comentario.dto';
import { Comentario } from './entities/comentario.entity';
import { User } from 'src/shared/users/entities/user.entity';
import { PostInteration } from './entities/postInteration.entity';
import { Roles } from 'src/shared/enums/enum';
import { CourseService } from 'src/ecommerce/courses/courses.service';
import { AgendaService } from '../agenda/agenda.service';

@Injectable()
export class PostService {
  constructor(
    @InjectRepository(Post)
    private postRepository: Repository<Post>,
    @InjectRepository(Comentario)
    private comentarioRepository: Repository<Comentario>,
    @InjectRepository(PostInteration)
    private postInterationRepository: Repository<PostInteration>,
    private readonly courseService: CourseService,
    private readonly agendaService: AgendaService,
  ) {}
  private s3 = new S3Client({
    region: process.env.AWS_S3_REGION!,
    credentials: {
      accessKeyId: process.env.AWS_ACCESS_KEY_ID!,
      secretAccessKey: process.env.AWS_SECRET_ACCESS_KEY!,
    },
  });

  async create(
    createPostDto: CreatePostDto,
    currentUser: User,
    imagem?: Express.Multer.File,
  ) {
    let imageUrl: string | undefined;

    if (imagem) {
      const fileName = `posts/${Date.now()}-${imagem.originalname}`;
      imageUrl = await this.uploadImageToS3(imagem, fileName);
    }

    const post = this.postRepository.create({
      titulo: createPostDto.titulo,
      subtitulo: createPostDto.subtitulo,
      corpo: createPostDto.corpo,
      tipo: createPostDto.tipo,
      data: new Date(),
      url: imageUrl,
      user: {
        id: currentUser.id,
      },
      categoria: createPostDto.categoria,
    });

    return await this.postRepository.save(post);
  }

  async findAll(currentUser: User) {
    const all = await this.postRepository
      .createQueryBuilder('post')
      .leftJoinAndSelect('post.user', 'user')

      // TOTAL DE COMENTÁRIOS VÁLIDOS
      .loadRelationCountAndMap(
        'post.totalComentarios',
        'post.comentario',
        'comentario',
        (qb) =>
          qb
            .where('comentario.corpo IS NOT NULL')
            .andWhere('comentario.corpo <> :empty', { empty: '' }),
      )

      // TOTAL DE VIEWS
      .loadRelationCountAndMap(
        'post.totalViews',
        'post.postInteration',
        'postInterationView',
        (qb) => qb.where('postInterationView.viewPost = :view', { view: true }),
      )

      // TOTAL DE LIKES
      .loadRelationCountAndMap(
        'post.totalLikes',
        'post.postInteration',
        'postInterationLike',
        (qb) => qb.where('postInterationLike.likePost = :like', { like: true }),
      )

      // TOTAL DE DESLIKES
      .loadRelationCountAndMap(
        'post.totalDeslikes',
        'post.postInteration',
        'postInterationDeslike',
        (qb) =>
          qb.where('postInterationDeslike.deslikePost = :deslike', {
            deslike: true,
          }),
      )

      .select([
        'post.id',
        'post.titulo',
        'post.subtitulo',
        'post.corpo',
        'post.categoria',
        'post.data',
        'post.url',
        'post.tipo',
        'user.user_name',
        'user.email',
      ])
      .getMany();

    if (currentUser.roles === Roles.USER) {
      const courses = await this.courseService.findAllByUser(currentUser.id);

      const courseIds = courses.map((course) => course.id);

      const courseProgressList = await Promise.all(
        courseIds.map(async (courseId) => {
          try {
            return await this.courseService.courseProgress(
              currentUser.id,
              courseId,
            );
          } catch (error) {
            // Ignora cursos que o usuário não é inscrito
            return null;
          }
        }),
      );

      // Remove os nulls (cursos não inscritos)
      const validCourseProgress = courseProgressList.filter(Boolean);

      const agendamentos = await this.agendaService.findAll();
      return {
        posts: all,
        course: courses,
        courseProgress: validCourseProgress,
        agendamentos: agendamentos,
      };
    }

    return all;
  }

  async findOne(id: number, currentUser: User) {
    const existPost = await this.postRepository.findOne({
      where: { id },
    });

    if (!existPost) {
      throw new BadRequestException('Post não existe!');
    }

    const isview = await this.postInterationRepository.findOne({
      where: { post: { id }, user: { id: currentUser.id } },
    });

    if (isview === null) {
      const create = await this.postInterationRepository.create({
        user: { id: currentUser.id },
        viewPost: true,
        post: { id: id },
      });
      await this.postInterationRepository.save(create);
    }

    if (isview?.viewPost === false) {
      await this.postInterationRepository.update(
        { user: { id } },
        { viewPost: true },
      );
    }

    const post = await this.postRepository
      .createQueryBuilder('post')
      .innerJoinAndSelect('post.user', 'user')
      .loadRelationCountAndMap(
        'post.totalLikes',
        'post.postInteration',
        'postInteration',
        (qb) => qb.where('postInteration.likePost = :like', { like: true }),
      )
      .loadRelationCountAndMap(
        'post.totalDeslikes',
        'post.postInteration',
        'postInteration',
        (qb) =>
          qb.where('postInteration.deslikePost = :deslike', { deslike: true }),
      )
      .loadRelationCountAndMap(
        'post.view',
        'post.postInteration',
        'postInteration',
        (qb) => qb.where('postInteration.viewPost = :view', { view: true }),
      )
      .select([
        'post.id',
        'post.titulo',
        'post.subtitulo',
        'post.corpo',
        'post.data',
        'post.like',
        'post.url',
        'user.id',
        'user.user_name',
        'user.email',
      ])
      .where('post.id = :id', { id })
      .getOne();

    if (!post) return null;

    const comentarios = await this.comentarioRepository
      .createQueryBuilder('comentario')
      .innerJoinAndSelect('comentario.user', 'user')
      .innerJoin('comentario.post', 'post')
      .select([
        'comentario.corpo',
        'comentario.data',
        'comentario.like',
        'comentario.id',
        'comentario.respostaId',
        'user.id',
        'user.user_name',
        'user.email',
      ])
      .where('post.id = :id', { id })
      .andWhere('comentario.corpo IS NOT NULL')
      .andWhere('comentario.corpo <> :empty', { empty: '' })
      .getRawMany();

    const interacaoPost = await this.postInterationRepository
      .createQueryBuilder('postinteration')
      .innerJoin('postinteration.post', 'post')
      .innerJoin('postinteration.user', 'user')
      .select([
        'postinteration.viewPost AS viewPost',
        'postinteration.likePost AS likePost',
        'postinteration.deslikePost AS deslikePost',
      ])
      .where('post.id = :postId', { postId: id })
      .andWhere('user.id =  :userId', { userId: currentUser.id })
      .getRawMany();

    const resultado = {
      ...post,
      comentarios,
      interacaoPost,
    };

    return resultado;
  }

  private buildTree(comentarios: Comentario[]) {
    const map = new Map<number, any>();
    const roots: any[] = [];

    comentarios.forEach((c) => {
      map.set(c.id, {
        ...c,
        respostas: [],
      });
    });

    comentarios.forEach((c) => {
      if (c.resposta?.id) {
        const pai = map.get(c.resposta.id);
        if (pai) {
          pai.respostas.push(map.get(c.id));
        }
      } else {
        roots.push(map.get(c.id));
      }
    });

    return roots;
  }

  async getComentarios(postId: number) {
    const comentarios = await this.comentarioRepository
      .createQueryBuilder('comentario')
      .leftJoinAndSelect('comentario.user', 'user')
      .leftJoin('comentario.resposta', 'resposta') // 👈 join no pai
      .where('comentario.postId = :postId', { postId })
      .select([
        'comentario.id',
        'comentario.corpo',
        'comentario.data',
        'comentario.like',
        'user.user_name',
        'user.email',
        'resposta.id',
      ])
      .orderBy('comentario.data', 'ASC')
      .getMany();

    return this.buildTree(comentarios);
  }

  async like(id: number, currentUser: User) {
    const existPost = await this.postRepository.findOne({
      where: { id },
    });

    if (!existPost) {
      throw new BadRequestException('Post não existe!');
    }
    const isdeslike = await this.postInterationRepository.findOne({
      where: {
        post: { id: id },
        user: { id: currentUser.id },
        deslikePost: true,
      },
    });

    if (isdeslike) {
      await this.postInterationRepository.update(
        { user: { id: currentUser.id }, post: { id: id } },
        {
          likePost: true,
          deslikePost: false,
        },
      );
    }

    await this.postInterationRepository.update(
      { user: { id: currentUser.id }, post: { id: id } },
      {
        likePost: true,
      },
    );

    return 'like adicionado!';
  }

  async deslike(id: number, currentUser: User) {
    const existPost = await this.postRepository.findOne({
      where: { id },
    });

    if (!existPost) {
      throw new BadRequestException('Post não existe!');
    }

    const islike = await this.postInterationRepository.findOne({
      where: {
        post: { id: id },
        user: { id: currentUser.id },
        likePost: true,
      },
    });

    if (islike) {
      await this.postInterationRepository.update(
        { user: { id: currentUser.id }, post: { id: id } },
        {
          deslikePost: true,
          likePost: false,
        },
      );
    }

    await this.postInterationRepository.update(
      { user: { id: currentUser.id }, post: { id: id } },
      {
        deslikePost: true,
      },
    );

    return 'deslike adicionado!';
  }

  async createComentario(
    createComentarioDto: CreateComentarioDto,
    currentUser: User,
    postId: number,
  ) {
    const newComentario = this.comentarioRepository.create({
      corpo: createComentarioDto.corpo,
      data: new Date(),
      post: { id: postId },
      user: { id: currentUser.id },

      // 🔽 se for resposta, aponta para o comentário pai
      resposta: createComentarioDto.respostaId
        ? { id: createComentarioDto.respostaId }
        : undefined,
    });

    return await this.comentarioRepository.save(newComentario);
  }

  async update(id: number, updatePostDto: UpdatePostDto) {
    const exist = await this.postRepository.findOne({
      where: { id },
    });

    if (!exist) {
      throw new BadRequestException('Post não existe');
    }

    return await this.postRepository.update(
      { id },
      {
        ...updatePostDto,
      },
    );
  }

  async remove(id: number) {
    const exist = await this.postRepository.findOne({
      where: { id },
    });

    if (!exist) {
      throw new BadRequestException('Post não existe');
    }

    return await this.postRepository.delete(id);
  }

  async removeComentario(idComentario: number) {
    const exist = await this.comentarioRepository.findOne({
      where: { id: idComentario },
    });

    if (!exist) {
      throw new BadRequestException('Comentario não existe');
    }

    return await this.comentarioRepository.delete(idComentario);
  }

  async uploadImageToS3(
    file: Express.Multer.File,
    fileName: string,
  ): Promise<string> {
    const bucketName = process.env.AWS_S3_BUCKET!;

    //  const buffer = await sharp(file?.buffer)

    //         .toBuffer();

    const uploadParams = new PutObjectCommand({
      Bucket: bucketName,
      Key: fileName,
      Body: file.buffer,
      ContentType: file.mimetype,
    });

    try {
      await this.s3.send(uploadParams);
      return `https://${bucketName}.s3.${process.env.AWS_S3_REGION}.amazonaws.com/${fileName}`;
    } catch (err) {
      throw new BadRequestException('Erro ao enviar arquivo para o S3: ' + err);
    }
  }
}
