import { BadRequestException, ConflictException, Injectable, NotFoundException } from '@nestjs/common';
import { CreateWatcherDto } from './dto/create-watcher.dto';
import { InjectRepository } from '@nestjs/typeorm';
import { Watcher } from './entities/watcher.entity';
import { Repository } from 'typeorm';
import { User } from 'src/shared/users/entities/user.entity';

@Injectable()
export class WatchersService {
  constructor(
    @InjectRepository(Watcher)
    private watcherRepository: Repository<Watcher>,

    @InjectRepository(User)
    private userRepository: Repository<User>,

  ) {}
  async create(id: number, createWatcherDto: CreateWatcherDto) {
    const user = await this.userRepository.findOneBy({ email: createWatcherDto.email});

    if (!user) {
      throw new NotFoundException("Usuário não encontrado pelo e-mail informado");
    }

    const watcherExist = await this.watcherRepository.findOne({
      where: {
        user_owner: {id},
        user_watcher: { id: user.id } 
      }
    })

    if (watcherExist) {
      throw new ConflictException("Obsevador já atrelado ao proprieário")
    }
    
    const watcher = this.watcherRepository.create({
      user_owner: { id },
      user_watcher: { id: user.id }
    })

    return this.watcherRepository.save(watcher);
  }

  async findAllByOwner(id: number) {
    const user = await this.userRepository.findOneBy({id});

    if (!user) { 
      throw new NotFoundException("Usuário não encontrado");
    }

    return await this.watcherRepository
           .createQueryBuilder('watcher')
           .innerJoin('watcher.user_owner', 'userOwner')
           .innerJoin('watcher.user_watcher', 'userWatcher')
           .select([
            'watcher.id AS id',
            'userWatcher.document AS document',
            'userWatcher.company_name AS company_name',
            'userWatcher.user_name AS user_name',
            'userWatcher.email AS email',
            'userWatcher.cellphone AS cellphone'
           ])
           .where('owner_id = :id', {id})
           .getRawMany();  
  }

  async findAllByWatcher(id: number) {
    const user = this.userRepository.findOneBy({id});

    if (!user) {
      throw new NotFoundException("Usuário não encontrado");
    }

    try{
      return await this.watcherRepository
      .createQueryBuilder('watcher')
      .innerJoinAndSelect('watcher.user_owner', 'ownerUser')
      .select([
        'watcher.id AS id',
        'ownerUser.company_name AS company_name',
        'ownerUser.user_name AS user_name' 
      ])
      .where('watcher_id = :id', {id})
      .getRawMany();
    } catch (error) {
      throw new BadRequestException("Não foi possível retornar a de usuários assistidos");
    }
  }

  async remove(id: number, watcherId: number) {
    const user = await this.userRepository.findOneBy({id});
    const watcher = await this.watcherRepository.findOne({
      where: [
        {
        id: watcherId,
        user_owner: {id}
        },
        {
          id: watcherId,
          user_watcher: {id}
        }
      ]
    })

    if (!user || !watcher) {
      throw new NotFoundException("Usuário não encontrado");
    }

    return await this.watcherRepository.remove(watcher);
  }
}
